﻿/* 
 * Copyright (C) 2019 Asseco Central Europe, a.s. All rights reserved. 
 * 
 * Resources:
 * https://docs.microsoft.com/en-us/dotnet/standard/security/how-to-sign-xml-documents-with-digital-signatures
 * https://www.dasm.cz/clanek/podepisovani-xml-sha256-hashi-v-net
 * https://blogs.msdn.microsoft.com/shawnfa/2005/11/03/adding-signatureproperties-to-signedxml/
 * https://stackoverflow.com/questions/50096199/c-sharp-how-to-properly-sign-message-with-xades-using-signedxml
 * 
 * And Microsoft France team Xades implementation:
 * https://github.com/wjab/xadesnet
 */

using System;
using System.Text;
using System.Xml;
using System.IO;

namespace XAdES_BES
{
    class Program
    {
        static void Main(string[] args)
        {
            string document = Properties.Settings.Default.docin;
            string documentSigned = Properties.Settings.Default.docout;
            string certificate = Properties.Settings.Default.cert;
            string password = Properties.Settings.Default.pass;

            string elementToSign = Properties.Settings.Default.elementToSign;
            string elementNSToSign = Properties.Settings.Default.elementNSToSign;

            string defaultSigningTime = Properties.Settings.Default.signingTime;

            string signatureId = Properties.Settings.Default.signatureId;

            bool useSHA1 = string.IsNullOrWhiteSpace(Properties.Settings.Default.method) ? true : Properties.Settings.Default.method.ToLower().Equals("sha1");

            if (string.IsNullOrWhiteSpace(document)
                || string.IsNullOrWhiteSpace(documentSigned)
                || string.IsNullOrWhiteSpace(certificate)
                || string.IsNullOrWhiteSpace(password))
            {
                log("must insert all parameters(docin, docout, cert, pass)");
                Environment.Exit(1);
            }

            if (!CheckFile(document) || !CheckFile(certificate))
            {
                Environment.Exit(1);
            }

            try
            {
                signBes(document, documentSigned, certificate, password, useSHA1, elementToSign, elementNSToSign, defaultSigningTime, signatureId);
            }
            catch (Exception e)
            {
                log("EXCEPTION");
                log(e.Message);
                log(e.StackTrace);
            }

            Console.WriteLine("Done");
        }

        public static void signBes(string document, string documentSigned, string certificate, string password, bool useSHA1, string elementToSign, string elementNSToSign = null, string defaultSigningTime = null, string signatureId = null)
        {
            // Load an XML file into the XmlDocument object.
            XmlDocument doc = new XmlDocument();
            doc.PreserveWhitespace = true;
            doc.Load(document);

            // Sign document
            SignUtils su = new SignUtils();
            XmlDocument signed_doc = su.signBes(doc, certificate, password, useSHA1, elementToSign, elementNSToSign, defaultSigningTime, signatureId);

            if (signed_doc != null)
            {
                log("XML file signed.");
                log("For verification can be used https://tools.chilkat.io/xmlDsigVerify.cshtml");

                //Add XML declaration to the document.
                //signed_doc.InsertBefore(signed_doc.CreateXmlDeclaration("1.0", "UTF-8", "no"), signed_doc.DocumentElement);

                if (Properties.Settings.Default.useutf8writer)
                {
                    // Save the document.
                    using (var writer = new XmlTextWriter(documentSigned, new UTF8Encoding(false)))
                    {
                        signed_doc.Save(writer);
                    }
                }
                else
                {
                    signed_doc.Save(documentSigned);
                }
            }
            else
            {
                log("XML file NOT signed.");
            }
        }

        public static void log(String text)
        {
            Console.WriteLine(text);
            System.Diagnostics.Debug.WriteLine(text);
        }

        private static bool CheckFile(string fileName)
        {
            if (!File.Exists(fileName))
            {
                log(string.Format("File {0} not found!", fileName));
                return false;
            }

            return true;
        }
    }
}
