﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Numerics;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Security.Cryptography.Xml;
using System.Xml;

namespace XAdES_BES
{
    public class SignUtils
    {
        public XmlDocument signBes(XmlDocument doc, string keyStorePath, string keyStorePassword, bool useSHA1, string elementToSign, string elementNSToSign = null, string defaultSigningTime = null, string signatureId = null)
        {
            XmlElement elemToSign = null;

            // Signing element olny?
            if (!string.IsNullOrWhiteSpace(elementToSign))
            {
                var nodeList = string.IsNullOrWhiteSpace(elementNSToSign) ? doc.SelectNodes("//*[local-name()='" + elementToSign + "']") : doc.GetElementsByTagName(elementToSign, elementNSToSign);
                if (nodeList.Count == 0)
                {
                    throw new Exception("No element '" + elementToSign + "' was found");
                }

                elemToSign = (XmlElement)nodeList[0];
            }

            // Read certificate
            X509Certificate2 certificate = new X509Certificate2(keyStorePath, keyStorePassword);

            // Get private key
            // could be RSACryptoServiceProvider or RSACng 
            var privateKey = certificate.GetRSAPrivateKey();

            // Sign the XML document. 
            SignXml(doc, elemToSign, privateKey, certificate, useSHA1, defaultSigningTime, signatureId);
            return doc;
        }

        // Sign an XML file. 
        // This document cannot be verified unless the verifying 
        // code has the key with which it was signed.
        public static void SignXml(XmlDocument doc, XmlElement elemToSign, RSA privateKey, X509Certificate2 certificate, bool useSHA1 = true, string defaultSigningTime = null, string signatureId = null)
        {
            // Check arguments.
            if (doc == null)
                throw new ArgumentException(nameof(doc));
            if (privateKey == null)
                throw new ArgumentException(nameof(privateKey));

            // Required custom signature.Id?
            if (string.IsNullOrWhiteSpace(signatureId))
            {
                signatureId = "xmldsig-" + Guid.NewGuid().ToString();
            }

            // Create a SignedXml object.            
            XadesSignedXml signedXml = (elemToSign == null) ? new XadesSignedXml(doc) : new XadesSignedXml(elemToSign);
            signedXml.Signature.Id = signatureId;

            // Add the key to the SignedXml document.
            signedXml.SigningKey = privateKey;
            signedXml.SignedInfo.SignatureMethod = useSHA1 ? XadesSignedXml.XmlDsigRSASHA1Url : XadesSignedXml.XmlDsigRSASHA256Url;

            // Add ID to SignedValue
            signedXml.SignatureValueId = signedXml.Signature.Id + "-sigvalue";

            // Create a reference to be signed.  Pass "" to specify that all of the current XML document should be signed.
            Reference reference = new Reference();
            reference.Uri = "";
            reference.Id = signedXml.Signature.Id + "-ref0";

            // Add an enveloped transformation to the reference.           
            if (useSHA1)
            {
                reference.AddTransform(new XmlDsigEnvelopedSignatureTransform());
                reference.AddTransform(new XmlDsigC14NTransform());
                reference.DigestMethod = XadesSignedXml.XmlDsigSHA1Url;
            }
            else
            {
                reference.AddTransform(new XmlDsigEnvelopedSignatureTransform());
                reference.AddTransform(new XmlDsigExcC14NTransform(true));
                reference.DigestMethod = XadesSignedXml.XmlDsigSHA256Url;
            }

            // Add the reference to the SignedXml object.
            signedXml.AddReference(reference);

            // Pridani verejneho klice do objektu pro moznost kontroly podpisu prijemcem
            signedXml.KeyInfo = new KeyInfo();
            signedXml.KeyInfo.AddClause(new KeyInfoX509Data(certificate, X509IncludeOption.WholeChain));

            AddXAdESProperties(doc, signedXml, certificate, useSHA1, defaultSigningTime);

            // Compute the signature.
            signedXml.ComputeSignature();

            // Get the XML representation of the signature and save
            // it to an XmlElement object.
            XmlElement xmlDigitalSignature = signedXml.GetXml();

            // Where to add signature?
            if (elemToSign != null)
            {
                // Append signature after signed elelemnt
                elemToSign.AppendChild(doc.ImportNode(xmlDigitalSignature, true));
            }
            else
            {
                // Append the element to the XML document.
                doc.DocumentElement.AppendChild(doc.ImportNode(xmlDigitalSignature, true));
            }
        }

        private static void AddXAdESProperties(XmlDocument document, XadesSignedXml xadesSignedXml, X509Certificate2 signingCertificate, bool useSHA1 = true, string defaultSigningTime = null)
        {
            var parametersSignature = new Reference
            {
                Uri = "#" + xadesSignedXml.Signature.Id + "-signedprops",
                Type = XadesSignedXml.XmlDsigSignatureProperties,
                DigestMethod = useSHA1 ? XadesSignedXml.XmlDsigSHA1Url : XadesSignedXml.XmlDsigSHA256Url
            };

            // Is this necessary?
            //if (!useSHA1) parametersSignature.AddTransform(new XmlDsigC14NTransform()); 

            xadesSignedXml.AddReference(parametersSignature);

            // <Object>
            var objectNode = document.CreateElement("ds", "Object", XadesSignedXml.XmlDsigNamespaceUrl);

            // <Object><QualifyingProperties>
            var qualifyingPropertiesNode = document.CreateElement(XadesSignedXml.XadesPrefix, "QualifyingProperties", XadesSignedXml.XadesNamespaceUrl);
            qualifyingPropertiesNode.SetAttribute("Target", $"#{xadesSignedXml.Signature.Id}");
            objectNode.AppendChild(qualifyingPropertiesNode);

            // <Object><QualifyingProperties><SignedProperties>
            var signedPropertiesNode = document.CreateElement(XadesSignedXml.XadesPrefix, "SignedProperties", XadesSignedXml.XadesNamespaceUrl);
            signedPropertiesNode.SetAttribute("Id", xadesSignedXml.Signature.Id + "-signedprops");
            qualifyingPropertiesNode.AppendChild(signedPropertiesNode);

            // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties>
            var signedSignaturePropertiesNode = document.CreateElement(XadesSignedXml.XadesPrefix, "SignedSignatureProperties", XadesSignedXml.XadesNamespaceUrl);
            signedPropertiesNode.AppendChild(signedSignaturePropertiesNode);

            // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties> </SigningTime>
            // ISO 8601 format as required in http://www.w3.org/TR/xmlschema-2/#dateTime 
            var signingTime = document.CreateElement(XadesSignedXml.XadesPrefix, "SigningTime", XadesSignedXml.XadesNamespaceUrl);
            signingTime.InnerText = string.IsNullOrWhiteSpace(defaultSigningTime) ? DateTime.Now.ToString("yyyy-MM-ddTHH:mm:ss.fffzzz") : defaultSigningTime;
            signedSignaturePropertiesNode.AppendChild(signingTime);

            // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate>
            var signingCertificateNode = document.CreateElement(XadesSignedXml.XadesPrefix, "SigningCertificate", XadesSignedXml.XadesNamespaceUrl);
            signedSignaturePropertiesNode.AppendChild(signingCertificateNode);

            // Add certificates
            var keyInfoClause = xadesSignedXml.KeyInfo.GetEnumerator();
            if (keyInfoClause.MoveNext())
            {
                KeyInfoX509Data data = (KeyInfoX509Data)keyInfoClause.Current;

                foreach (X509Certificate2 certificate in data.Certificates)
                {
                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert>
                    var certNode = document.CreateElement(XadesSignedXml.XadesPrefix, "Cert", XadesSignedXml.XadesNamespaceUrl);
                    signingCertificateNode.AppendChild(certNode);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><CertDigest>
                    var certDigestNode = document.CreateElement(XadesSignedXml.XadesPrefix, "CertDigest", XadesSignedXml.XadesNamespaceUrl);
                    certNode.AppendChild(certDigestNode);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><CertDigest> </DigestMethod>
                    var digestMethod = document.CreateElement("DigestMethod", XadesSignedXml.XmlDsigNamespaceUrl);
                    var digestMethodAlgorithmAtribute = document.CreateAttribute("Algorithm");
                    digestMethodAlgorithmAtribute.InnerText = useSHA1 ? XadesSignedXml.XmlDsigSHA1Url : XadesSignedXml.XmlDsigSHA256Url;
                    digestMethod.Attributes.Append(digestMethodAlgorithmAtribute);
                    certDigestNode.AppendChild(digestMethod);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><CertDigest> </DigestMethod>
                    var digestValue = document.CreateElement("DigestValue", XadesSignedXml.XmlDsigNamespaceUrl);

                    if (useSHA1)
                    {
                        // SHA1 (.Net 4.8+ allow to specify HashAlgorithm for GetCertHash())
                        digestValue.InnerText = Convert.ToBase64String(certificate.GetCertHash());
                    }
                    else
                    {
                        // SHA256
                        using (HashAlgorithm hashAlgo = HashAlgorithm.Create(Hash​Algorithm​Name.SHA256.Name))
                        {
                            digestValue.InnerText = Convert.ToBase64String(hashAlgo.ComputeHash(certificate.RawData));
                        }
                    }

                    certDigestNode.AppendChild(digestValue);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><IssuerSerial>
                    var issuerSerialNode = document.CreateElement(XadesSignedXml.XadesPrefix, "IssuerSerial", XadesSignedXml.XadesNamespaceUrl);
                    certNode.AppendChild(issuerSerialNode);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><IssuerSerial> </X509IssuerName>
                    var x509IssuerName = document.CreateElement("X509IssuerName", XadesSignedXml.XmlDsigNamespaceUrl);
                    x509IssuerName.InnerText = certificate.Issuer;
                    issuerSerialNode.AppendChild(x509IssuerName);

                    // <Object><QualifyingProperties><SignedProperties><SignedSignatureProperties><SigningCertificate><Cert><IssuerSerial> </X509SerialNumber>
                    var x509SerialNumber = document.CreateElement("X509SerialNumber", XadesSignedXml.XmlDsigNamespaceUrl);
                    //x509SerialNumber.InnerText = ToDecimalString(certificate.SerialNumber);
                    x509SerialNumber.InnerText = certificate.SerialNumber;
                    issuerSerialNode.AppendChild(x509SerialNumber);
                }
            }

            /*
            // <Object><SignaturePolicyIdentifier><SignaturePolicyImplied /><SignaturePolicyIdentifier>
            var signaturePolicyIdentifier = document.CreateElement(XadesSignedXml.XadesPrefix, "SignaturePolicyIdentifier", XadesSignedXml.XadesNamespaceUrl);
            signedSignaturePropertiesNode.AppendChild(signaturePolicyIdentifier);
            var implied = document.CreateElement(XadesSignedXml.XadesPrefix, "SignaturePolicyImplied", XadesSignedXml.XadesNamespaceUrl);
            signaturePolicyIdentifier.AppendChild(implied);
            */

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties>
            var signedDataObjectPropertiesNode = document.CreateElement(XadesSignedXml.XadesPrefix, "SignedDataObjectProperties", XadesSignedXml.XadesNamespaceUrl);
            signedPropertiesNode.AppendChild(signedDataObjectPropertiesNode);

            /*
            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><CommitmentTypeIndication>
            var commitmentTypeIndicationNode = document.CreateElement(XadesSignedXml.XadesPrefix, "CommitmentTypeIndication", XadesSignedXml.XadesNamespaceUrl);
            signedDataObjectPropertiesNode.AppendChild(commitmentTypeIndicationNode);

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><CommitmentTypeIndication><CommitmentTypeId>
            var commitmentTypeIdNode = document.CreateElement(XadesSignedXml.XadesPrefix, "CommitmentTypeId", XadesSignedXml.XadesNamespaceUrl);
            commitmentTypeIndicationNode.AppendChild(commitmentTypeIdNode);

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><CommitmentTypeIndication><CommitmentTypeId><Identifier>
            var identifierNode = document.CreateElement(XadesSignedXml.XadesPrefix, "Identifier", XadesSignedXml.XadesNamespaceUrl);
            identifierNode.InnerText = XadesSignedXml.XadesProofOfApproval;
            commitmentTypeIdNode.AppendChild(identifierNode);

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><CommitmentTypeIndication><AllSignedDataObjects>
            var allSignedDataObjectsNode = document.CreateElement(XadesSignedXml.XadesPrefix, "AllSignedDataObjects", XadesSignedXml.XadesNamespaceUrl);
            commitmentTypeIndicationNode.AppendChild(allSignedDataObjectsNode);
            */

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><DataObjectFormat>
            var dataObjectFormatNode = document.CreateElement(XadesSignedXml.XadesPrefix, "DataObjectFormat", XadesSignedXml.XadesNamespaceUrl);
            XmlAttribute attr = document.CreateAttribute("ObjectReference");
            attr.Value = "#" + xadesSignedXml.Signature.Id + "-ref0";
            dataObjectFormatNode.Attributes.Append(attr);
            signedDataObjectPropertiesNode.AppendChild(dataObjectFormatNode);

            // <Object><QualifyingProperties><SignedProperties><SignedDataObjectProperties><DataObjectFormat><MimeType>
            var mimeTypeNode = document.CreateElement(XadesSignedXml.XadesPrefix, "MimeType", XadesSignedXml.XadesNamespaceUrl);
            mimeTypeNode.InnerText = "application/xml";
            dataObjectFormatNode.AppendChild(mimeTypeNode);

            // Set ds prefix and cache for later use
            xadesSignedXml.AddXadesObject(objectNode);

            // Add objet to Xml
            var dataObject = new DataObject();
            dataObject.Id = "XadesObject";
            dataObject.Data = qualifyingPropertiesNode.SelectNodes(".");
            xadesSignedXml.AddObject(dataObject);
        }

        private static string ToDecimalString(string serialNumber)
        {
            BigInteger bi;

            if (BigInteger.TryParse(serialNumber, NumberStyles.HexNumber, CultureInfo.InvariantCulture, out bi))
            {
                return bi.ToString(CultureInfo.InvariantCulture);
            }
            else
            {
                return serialNumber;
            }
        }
    }
}
