﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Xml;
using CSSZSubmissionDemo.Helpers;
using CSSZSubmissionDemo.Settings;

namespace CSSZSubmissionDemo.Submissions
{
    class WSResponse : SubmissionResponseBase
    {
        protected byte[] data;

        public byte[] Decode(ref WS.PollResponse data, out AckInfo ack)
        {
            this.data = Convert.FromBase64String(data.BodyBase64XML);
            ack = new AckInfo() { ID = "", Class = "", internalID = "", status = "", function = "", qualifier = "", sigValid = false };

            XmlDocument doc = new XmlDocument();
            doc.LoadXml(Encoding.UTF8.GetString(this.data));

            //correlationID
            ack.ID = data.CorrelationID;
            //function submit, delete
            ack.function = data.Function;
            //qualifier (request), ack, error, response
            ack.qualifier = data.Qualifier;

            //error
            if (data.GovTalkErrors != null && data.GovTalkErrors.Errors != null && data.GovTalkErrors.Errors.Count > 0)
            {
                ack.errCode = data.GovTalkErrors.Errors[0].Number.ToString();
                ack.errTxt = data.GovTalkErrors.Errors[0].Text;
            }

            //VREP Timestamp
            XmlNode nod = doc.SelectSingleNode("//*[local-name()='Signature' and namespace-uri()='http://www.w3.org/2000/09/xmldsig#']");
            if (nod != null)
            {
                //w3sig
                base.ValidateW3CSignature(ref nod);
            }
            nod = doc.SelectSingleNode("//*[local-name()='Signature' and namespace-uri()='http://www.podani.gov.cz/TxE/timestamp']");
            if (nod != null)
            {
                //ggsig
                nod = nod.OwnerDocument.DocumentElement;
                base.ValidateGGSignature(ref nod);
            }

            //data
            XmlNode xCSSZMessage = doc.SelectSingleNode("//*[local-name()='Message']");
            return DecodeCSSZMessage(ref ack, ref xCSSZMessage);
        }

        protected override string InternalDisplayName()
        {
            return "WSResponse";
        }
    }

    namespace WS
    {
        public class ServiceClient : ClientBase<IBusinessTransactions>, IBusinessTransactions
        {
            private SoapMessageListener listener;

            public ServiceClient(System.ServiceModel.Channels.Binding binding, System.ServiceModel.EndpointAddress remoteAddress, bool trackMessages = false) : base(binding, remoteAddress)
            {
                if (trackMessages)
                {
                    listener = new SoapMessageListener();
                }

                this.Endpoint.Behaviors.Add(listener);
            }

            #region IBusinessTransactions Members

            public SoapMessageListener Listener
            {
                get { return listener; }
            }

            public PollResponse Poll(string correlationid, BindingList<OptionalParameter> optionals)
            {
                listener.Reset();

                return this.Channel.Poll(correlationid, optionals);
            }

            public PollResponse Submit(string tclass, System.ComponentModel.BindingList<BodyPart> bodies, System.ComponentModel.BindingList<OptionalParameter> optionals)
            {
                listener.Reset();

                return this.Channel.Submit(tclass, bodies, optionals);
            }

            public void Dispose(string correlationid, BindingList<OptionalParameter> optionals)
            {
                listener.Reset();

                this.Channel.Dispose(correlationid, optionals);
            }

            public DataResponseBody DataRequest(string tclass, System.ComponentModel.BindingList<OptionalParameter> optionals)
            {
                return this.Channel.DataRequest(tclass, optionals);
            }

            #endregion
        }

        [ServiceContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public interface IBusinessTransactions
        {
            [OperationContract]
            [FaultContract(typeof(GGErrorException))]
            PollResponse Poll(string correlationid, System.ComponentModel.BindingList<OptionalParameter> optionals);

            [OperationContract]
            [FaultContract(typeof(GGErrorException))]
            PollResponse Submit(string tclass, System.ComponentModel.BindingList<BodyPart> bodies, System.ComponentModel.BindingList<OptionalParameter> optionals);

            [OperationContract]
            [FaultContract(typeof(GGErrorException))]
            void Dispose(string correlationid, System.ComponentModel.BindingList<OptionalParameter> optionals);

            [OperationContract]
            [FaultContract(typeof(GGErrorException))]
            DataResponseBody DataRequest(string tclass, System.ComponentModel.BindingList<OptionalParameter> optionals);
        }

        #region BodyPart
        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class BodyPart
        {
            [DataMember]
            public string Id;

            [DataMember]
            public XmlElement Body;
        }
        #endregion

        #region PollResponse
        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class PollResponse
        {
            string _CorrelationID;
            string _Class;
            string _Qualifier;
            string _Function;
            DateTime _GatewayTimestamp;
            string _BodyBase64XML;
            TimeSpan _PollInterval;
            GovTalkErrors _GovTalkErrors;

            [DataMember]
            public string CorrelationID
            {
                get { return _CorrelationID; }
                set { _CorrelationID = value; }
            }

            [DataMember]
            public string Class
            {
                get { return _Class; }
                set { _Class = value; }
            }

            [DataMember]
            public string Function
            {
                get { return _Function; }
                set { _Function = value; }
            }

            [DataMember]
            public string Qualifier
            {
                get { return _Qualifier; }
                set { _Qualifier = value; }
            }

            [DataMember]
            public DateTime GatewayTimestamp
            {
                get { return _GatewayTimestamp; }
                set { _GatewayTimestamp = value; }
            }

            [DataMember]
            public TimeSpan PollInterval
            {
                get { return _PollInterval; }
                set { _PollInterval = value; }
            }

            [DataMember]
            public GovTalkErrors GovTalkErrors
            {
                get { return _GovTalkErrors; }
                set { _GovTalkErrors = value; }
            }

            [DataMember]
            public string BodyBase64XML
            {
                get { return _BodyBase64XML; }
                set { _BodyBase64XML = value; }
            }
        }

        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class GovTalkErrors
        {
            private System.ComponentModel.BindingList<Error> _Errors;

            [DataMember]
            public System.ComponentModel.BindingList<Error> Errors
            {
                get { return _Errors; }
                set { _Errors = value; }
            }
        }

        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class Error
        {
            private int _Id;
            private int _Number;
            private string _Location;
            private string _RaisedBy;
            private string _Type;
            private DateTime _GatewayTimeStamp;
            private string _text;

            public Error(int id, int number, string location, string raisedby, string type, DateTime gatewaytimestamp, string text)
            {
                _Id = id;
                _Number = number;
                _Location = location;
                _RaisedBy = raisedby;
                _Type = type;
                _Number = number;
                _GatewayTimeStamp = gatewaytimestamp;
                _text = text;
            }

            public Error(string text)
            {
                _text = text;
            }

            [DataMember]
            public int Id
            {
                get { return _Id; }
                set { _Id = value; }
            }

            [DataMember]
            public DateTime GatewayTimeStamp
            {
                get { return _GatewayTimeStamp; }
                set { _GatewayTimeStamp = value; }
            }

            [DataMember]
            public int Number
            {
                get { return _Number; }
                set { _Number = value; }
            }

            [DataMember]
            public string Location
            {
                get { return _Location; }
                set { _Location = value; }
            }

            [DataMember]
            public string RaisedBy
            {
                get { return _RaisedBy; }
                set { _RaisedBy = value; }
            }

            [DataMember]
            public string Type
            {
                get { return _Type; }
                set { _Type = value; }
            }

            [DataMember]
            public string Text
            {
                get { return _text; }
                set { _text = value; }
            }
        }
        #endregion

        #region OptionalParameter
        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class OptionalParameter
        {
            private string _parName;
            private string _parType;
            private string _parValue;

            [DataMember]
            public string ParameterName
            {
                get
                {
                    return _parName;
                }
                set
                {
                    _parName = value;
                }
            }

            [DataMember(IsRequired = false)]
            public string ParameterType
            {
                get
                {
                    return _parType;
                }
                set
                {
                    _parType = value;
                }
            }

            [DataMember]
            public string ParameterValue
            {
                get
                {
                    return _parValue;
                }
                set
                {
                    _parValue = value;
                }
            }
        }
        #endregion

        #region GGErrorException
        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class GGErrorException
        {
            private int _Number;
            private string _Location;
            private string _RaisedBy;
            private string _Type;
            private DateTime _GatewayTimeStamp;
            private string _text;

            public GGErrorException(int number, string location, string raisedby, string type, DateTime gatewaytimestamp, string text)
            {
                _Number = number;
                _Location = location;
                _RaisedBy = raisedby;
                _Type = type;
                _Number = number;
                _GatewayTimeStamp = gatewaytimestamp;
                _text = text;
            }

            public GGErrorException(string text)
            {
                _text = text;
            }

            [DataMember]
            public DateTime GatewayTimeStamp
            {
                get { return _GatewayTimeStamp; }
                set { _GatewayTimeStamp = value; }
            }

            [DataMember]
            public int Number
            {
                get { return _Number; }
                set { _Number = value; }
            }

            [DataMember]
            public string Location
            {
                get { return _Location; }
                set { _Location = value; }
            }

            [DataMember]
            public string RaisedBy
            {
                get { return _RaisedBy; }
                set { _RaisedBy = value; }
            }

            [DataMember]
            public string Type
            {
                get { return _Type; }
                set { _Type = value; }
            }

            [DataMember]
            public string Text
            {
                get { return _text; }
                set { _text = value; }
            }
        }
        #endregion

        #region enums
        public enum enServiceError
        {
            GeneraleError = 0,
            AuthenticateError = 1
        }

        public enum enQualifier
        {
            poll = 0,
            request = 1
        }

        public enum enFunction
        {
            list = 0,
            submit = 1
        }
        #endregion

        #region DataRequest response
        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class DataResponseBody
        {
            string _SenderID;
            DateTime _StartTimeStamp;
            DateTime _EndTimeStamp;

            [DataMember]
            public string SenderID
            {
                get { return _SenderID; }
                set { _SenderID = value; }
            }

            [DataMember]
            public DateTime StartTimeStamp
            {
                get { return _StartTimeStamp; }
                set { _StartTimeStamp = value; }
            }

            [DataMember]
            public DateTime EndTimeStamp
            {
                get { return _EndTimeStamp; }
                set { _EndTimeStamp = value; }
            }

            List<StatusRecord> _Records;

            [DataMember]
            public List<StatusRecord> Rows
            {
                get { return _Records; }
                set { _Records = value; }
            }
        }

        [DataContract(Namespace = "http://www.government-gateway.cz/wcf/submission")]
        public class StatusRecord
        {
            string _Status;
            DateTime _TimeStamp;
            string _CorrelationID;

            [DataMember]
            public string CorrelationID
            {
                get { return _CorrelationID; }
                set { _CorrelationID = value; }
            }

            [DataMember]
            public DateTime TimeStamp
            {
                get { return _TimeStamp; }
                set { _TimeStamp = value; }
            }

            [DataMember]
            public string Status
            {
                get { return _Status; }
                set { _Status = value; }
            }
        }
        #endregion
    }
}
