﻿using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Security.Cryptography;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;
using System.Security.Cryptography.Xml;
using System.Text;
using System.Xml;
using CSSZSubmissionDemo.Settings;

namespace CSSZSubmissionDemo.Submissions
{
    abstract class SubmissionResponseBase : MessageBase
    {
        protected byte[] DecodeCSSZMessage(ref AckInfo ack, ref XmlNode xCSSZMessage)
        {
            XmlNode xProcessingResult = null;
            XmlNode xZpracovaniProtokol = null;
            XmlNode xProcessingResponse = null;
            XmlNode nod = null;

            //signature
            if (xCSSZMessage != null)
            {
                bool CSSZTimeStampValid = this.ValidateCSSZResponseSignature(xCSSZMessage.OuterXml);
                Trace.WriteLine(String.Format("CSSZ TimeStamp Valid: {0}", CSSZTimeStampValid));
                ack.sigValid = CSSZTimeStampValid;

                nod = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Header']/*[local-name()='Signature']/*[local-name()='TimeStamp']/*[local-name()='date']");
                if (nod != null) ack.sigDate = nod.InnerText;
                nod = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Header']/*[local-name()='Signature']/*[local-name()='TimeStamp']/*[local-name()='time']");
                if (nod != null) ack.sigTime = nod.InnerText;

                //body
                nod = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Body']/*[local-name()='Body']");
                xProcessingResult = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Body']/*[local-name()='ProcessingResult']");
                if (xProcessingResult != null)
                {
                    return Encoding.UTF8.GetBytes(xProcessingResult.OuterXml);
                }
                xZpracovaniProtokol = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Body']/*[local-name()='ZpracovaniProtokol']");
                if (xZpracovaniProtokol != null)
                {
                    return Encoding.UTF8.GetBytes(xZpracovaniProtokol.OuterXml);
                }
                xProcessingResponse = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Body']/*[local-name()='ProcessingResponse']");
                if (xProcessingResponse != null)
                {
                    string encAlg = "";
                    string comp = "";
                    string encoding = "";
                    string respvalue = "";

                    nod = xCSSZMessage.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Body']/*[local-name()='ProcessingResponse']/*[local-name()='Data']");
                    if (nod != null)
                    {
                        encAlg = nod.Attributes["encryptionAlgorithm"].Value;
                        comp = nod.Attributes["compression"].Value;
                        encoding = nod.Attributes["contentEncoding"].Value;
                        respvalue = nod.InnerText;
                    }
                    if (string.IsNullOrWhiteSpace(respvalue) == false)
                    {
                        byte[] decoded = Convert.FromBase64String(respvalue);
                        byte[] decrypted = this.Decrypt(ref decoded);
                        byte[] decompressed = this.Decompress(ref decrypted);

                        return decompressed;
                    }
                }
            }
            return null;
        }

        #region HELPER METHODS
        protected byte[] Decompress(ref byte[] data)
        {
            if (data.Length < 1) return new byte[0];

            Trace.WriteLine("Decompressing");
            using (MemoryStream ms = new MemoryStream(data))
            {
                using (GZipStream gz = new GZipStream(ms, CompressionMode.Decompress))
                {
                    using (MemoryStream dec = new MemoryStream())
                    {
                        int iBuffSize = 2048;
                        byte[] buff = new byte[iBuffSize];
                        int iSizeRead = gz.Read(buff, 0, iBuffSize);
                        while (iSizeRead > 0)
                        {
                            dec.Write(buff, 0, iSizeRead);
                            iSizeRead = gz.Read(buff, 0, iBuffSize);
                        }
                        if (dec.Length > 0)
                        {
                            return dec.ToArray();
                        }
                        else
                        {
                            return new byte[0];
                        }

                    }
                }
            }
        }

        protected byte[] Decrypt(ref byte[] data)
        {
            if (data.Length < 1) return new byte[0];
            Trace.WriteLine("Decrypting");
            EnvelopedCms cms = new EnvelopedCms();
            cms.Decode(data);

            try
            {
                cms.Decrypt();
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception while decrypting: {0}", ex.ToString()));
                return new byte[0];
            }

            return cms.ContentInfo.Content;
        }


        protected byte[] Decode64(string b64data)
        {
            Trace.WriteLine("Decoding");
            return Convert.FromBase64String(b64data);
        }

        // Podepsana casova znacka odpovedi CSSZ
        protected bool ValidateCSSZResponseSignature(string content)
        {
            bool retval = true;
            XmlDocument xDoc = new XmlDocument();
            xDoc.LoadXml(content);
            XmlNode nod = xDoc.SelectSingleNode("//*[local-name()='Message' and namespace-uri()='http://www.cssz.cz/XMLSchema/envelope']");

            string signatureDigestAlg = "";
            signatureDigestAlg = xDoc.SelectSingleNode("//*[local-name()='Message']/*[local-name()='Header']/*[local-name()='Signature']/*[local-name()='DigestMethod']").Attributes["Algorithm"].Value;

            Trace.WriteLine("Read signature");
            XmlNode signatureNode = xDoc.SelectSingleNode("//*[local-name()='Message' and namespace-uri()='http://www.cssz.cz/XMLSchema/envelope']/*[local-name()='Header']/*[local-name()='Signature']/*[local-name()='SignatureValue']");
            string signatureValue = signatureNode.InnerText.Replace("\n", "").Replace("\r", "").Replace("\t", "");
            Trace.WriteLine("Remove value");
            signatureNode.InnerText = "";

            Trace.WriteLine("Canonicalize");
            XmlDocument txDoc = new XmlDocument();
            txDoc.LoadXml(nod.OuterXml);
            XmlDsigC14NTransform t = new XmlDsigC14NTransform();
            t.LoadInput(txDoc);

            Trace.WriteLine("Get hash");
            Byte[] hash = null;
            if (signatureDigestAlg == "http://www.w3.org/2000/09/xmldsig#sha1")
            {
                hash = t.GetDigestedOutput(new SHA1Managed());
            }
            else if (signatureDigestAlg == "http://www.w3.org/2001/04/xmlenc#sha512")
            {
                hash = t.GetDigestedOutput(new SHA512Managed());
            }


            Trace.WriteLine("Validate signature");
            SignedCms signed = new SignedCms();
            signed.Decode(Convert.FromBase64String(signatureValue));
            try
            {
                signed.CheckSignature(true);
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception while checking signature: {0}", ex.ToString()));
                retval = false;
            }

            Byte[] data = signed.ContentInfo.Content;
            Trace.WriteLine("Compare hash bytes");
            for (int i = 0; i < hash.Length; i++)
            {
                Trace.WriteLine(String.Format("Compare({3}) {0} to {1} = {2}", hash[i], data[i], (hash[i] == data[i]), i));
                if (hash[i] != data[i]) retval = false;
            }

            return retval;
        }

        //Podepsana casova znacka dorucenky VREP v proprietarnim formatu VREP
        protected bool ValidateGGSignature(ref XmlNode signedNode)
        {
            try
            {
                XmlNode nodSignatureValue = signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='SignatureValue']");
                XmlNode nodTimestamp = signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='TimeStamp']");

                XmlNode nodDate = signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='TimeStamp']/*[local-name()='date']");
                XmlNode nodTime = signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='TimeStamp']/*[local-name()='time']");

                DateTime dt = DateTime.ParseExact(nodDate.InnerText + " " + nodTime.InnerText, "yyyyMMdd HH:mm:ss", null);

                Trace.WriteLine("GGSig validation - signed date " + dt.ToShortDateString() + " " + dt.ToShortTimeString());

                XmlDocument docSigned = new XmlDocument();
                docSigned.LoadXml(signedNode.SelectSingleNode("//*[local-name()='Body']").OuterXml);

                XmlNode nodSignature = docSigned.SelectSingleNode("//*[local-name()='Signature']");
                XmlNode nodSignatureValueToRemove = docSigned.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='SignatureValue']");
                //Remove node SignatureValue
                nodSignature.RemoveChild(nodSignatureValueToRemove);

                //make correct canonization
                string serial = docSigned.DocumentElement.OuterXml;
                serial = serial.Replace("xmlns=\"http://www.podani.gov.cz/TxE/timestamp\"", "XVersionX");
                serial = serial.Replace("Version=\"1.0\"", "xmlns=\"http://www.podani.gov.cz/TxE/timestamp\"");
                serial = serial.Replace("XVersionX", "Version=\"1.0\"");

                string sXml = VREPStripWhitespaces(serial);
                byte[] bSigned = Encoding.UTF8.GetBytes(sXml);

                //compute sha1 hash
                SHA1Managed sha1 = new SHA1Managed();
                sha1.ComputeHash(bSigned);
                string shaInHex = "";
                foreach (int ii in sha1.Hash) shaInHex += String.Format("{0:x2}", ii).ToUpper();
                byte[] bHashComupted = System.Text.Encoding.UTF8.GetBytes(shaInHex);

                // Create a new, nondetached SignedCms message.
                byte[] encodedMessage = Convert.FromBase64String(nodSignatureValue.InnerText);
                SignedCms signedCms = new SignedCms();
                signedCms.Decode(encodedMessage);
                byte[] bHashDecoded = signedCms.ContentInfo.Content;

                X509Certificate2 signerCetrificate = signedCms.Certificates[signedCms.Certificates.Count - 1];
                Trace.WriteLine("GGSig validation signer " + signerCetrificate.Subject + " " + signerCetrificate.Issuer);
                try
                {
                    signedCms.CheckSignature(true);

                    if (bHashDecoded.Equals(bHashComupted) == true)
                    {
                        Trace.WriteLine("GGSig signature valid");
                        return true;
                    }
                    else
                    {
                        Trace.WriteLine("GGSig signature notvalid");
                        return true;
                    }
                }
                catch (Exception ex)
                {
                    Trace.WriteLine(string.Format("Exception while checking ggsig signature: {0}", ex.ToString()));
                    return false;
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception while checking ggsig: {0}", ex.ToString()));
                return false;
            }
        }

        //Podepsana casova znacka dorucenky VREP ve formatu xmldsig
        protected bool ValidateW3CSignature(ref XmlNode signedNode)
        {
            if (signedNode != null)
            {
                XmlNode nodCert = signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='Object']/*[local-name()='SignatureProperties' and @Id='gg.x509']/*[local-name()='SignatureProperty']/*[local-name()='SignerCertificate']");
                X509Certificate2 signerCert = new X509Certificate2();
                signerCert.Import(Convert.FromBase64String(nodCert.InnerText.ToString()));

                Trace.WriteLine("XMLDSIG timestamp " + signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='Object']/*[local-name()='SignatureProperties' and @Id='gg.properties']/*[local-name()='SignatureProperty']/*[local-name()='TimeStamp']").InnerText);
                Trace.WriteLine("XMLDSIG correlationID " + signedNode.SelectSingleNode("//*[local-name()='Signature']/*[local-name()='Object']/*[local-name()='SignatureProperties' and @Id='gg.properties']/*[local-name()='SignatureProperty']/*[local-name()='CorrelationID']").InnerText);

                try
                {
                    Trace.WriteLine("XMLDSIG signer " + signerCert.Subject + " " + signerCert.Issuer);

                    bool result = false;
                    XmlDocument docSigned = new XmlDocument();
                    docSigned.LoadXml(signedNode.OuterXml);

                    SignedXml sxml = new SignedXml(docSigned);
                    XmlNodeList nodes = docSigned.GetElementsByTagName("Signature");
                    XmlElement eSig = (XmlElement)nodes[0];
                    sxml.LoadXml(eSig);
                    if (eSig.GetElementsByTagName("KeyInfo").Count == 0)
                    {
                        AsymmetricAlgorithm key = (RSACryptoServiceProvider)signerCert.PublicKey.Key;
                        result = sxml.CheckSignature(key);
                    }
                    else
                    {
                        result = sxml.CheckSignature();
                    }

                    if (result)
                    {
                        Trace.WriteLine("XMLDSIG signature valid");
                        return result;
                    }
                    else
                    {
                        Trace.WriteLine("XMLDSIG signature invalid");
                        return result;
                    }
                }
                catch (Exception ex)
                {
                    Trace.WriteLine(string.Format("Exception while checking xmldsig: {0}", ex.ToString()));
                    return false;
                }
            }

            return false;
        }

        private static string VREPStripWhitespaces(string xml)
        {
            int beg, end, idx = 0;
            while ((idx = xml.IndexOf(">", idx)) != -1)
            {
                if (xml[idx - 1] == '/')
                {
                    end = idx - 2;
                    beg = end;
                    while (Char.IsWhiteSpace(xml[beg])) beg--;
                    xml = xml.Remove(beg + 1, end - beg);
                    idx = beg + 2;
                }
                beg = idx + 1;
                do
                {
                    idx++;
                } while (idx < xml.Length && Char.IsWhiteSpace(xml[idx]));
                if ((idx - beg) > 0)
                {
                    xml = xml.Remove(beg, idx - beg);
                    idx = beg;
                }
            }
            return xml;
        }

        #endregion
    }
}
