﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using System.Xml.Serialization;
using CSSZSubmissionDemo.Configuration;

namespace CSSZSubmissionDemo.DAL
{
    public enum SubmissionStatus
    {
        created,
        sent,
        finished
    }

    public class SubmissionRecord
    {
        public SubmissionRecord()
        {
        }

        public SubmissionRecord(Settings.AckInfo record, string channelDisplayName)
        {
            this.ID = record.ID;
            this.Class = record.Class;
            this.VS = record.VS;
            this.DateCreated = DateTime.Now;
            this.DateLastChanged = DateTime.Now;
            this.InternalID = record.internalID;
            this.ChannelDisplayName = channelDisplayName;
        }

        public string ID { get; set; }
        public string Class { get; set; }
        public string InternalID { get; set; }
        public string VS { get; set; }
        public DateTime DateCreated { get; set; }
        public string ChannelDisplayName { get; set; }
        public SubmissionStatus Status { get; set; }
        public DateTime DateLastChanged { get; set; }
    }

    public class SubmissionDetails
    {
        public SubmissionDetails()
        {
        }

        public SubmissionDetails(string file)
        {
            this.FilePath = file;
            Init(this.FilePath);
        }

        public string Name { get; private set; }

        public DateTime CreationTime { get; private set; }
        public DateTime ModificationTime { get; private set; }

        [System.ComponentModel.Browsable(false)]
        public string FilePath { get; private set; }

        private void Init(string file)
        {
            if (File.Exists(file) == true)
            {
                FileInfo fi = new FileInfo(file);
                this.CreationTime = fi.CreationTime;
                this.ModificationTime = fi.LastWriteTime;
                this.Name = Path.GetFileNameWithoutExtension(fi.Name);
            }
        }
    }

    // Pomocna jednoducha trida pro ulozeni informaci o podanich do souboroveho systemu pro ucely testovani, v produkcni aplikaci nahradte transakcnim ulozistem
    public static class Connection
    {
        private static object lockObj = new object();

        private const string INDEX_FILE = "index.xml";
        private const string DATA_FILE_EXT = "txt";
        private const string SUBMISSIONS_FOLDER = "CSSZSubmissionData";

        private static string GetSubmissionFilePath(string internalID, string dataID)
        {
            string folder = GetSubmissionFolderPath(internalID);
            string f = "";
            if (dataID == INDEX_FILE)
            {
                f = Path.Combine(folder, dataID);
            }
            else if (string.IsNullOrWhiteSpace(dataID) == false)
            {
                f = Path.Combine(folder, string.Format("{0:yyyyMMdd-HHmmssfff}-{1}.{2}", DateTime.Now, dataID, DATA_FILE_EXT));
            }
            else
            {
                f = Path.Combine(folder, string.Format("{0:yyyyMMdd-HHmmssfff}.{2}", DateTime.Now, dataID, DATA_FILE_EXT));
            }

            return f;
        }

        private static string GetSubmissionFolderPath(string internalID)
        {
            string submissionsPath = GetSubmissionsFolderPath();
            string fullPath = Path.Combine(submissionsPath, internalID);
            if (Directory.Exists(fullPath) == false) Directory.CreateDirectory(fullPath);
            return fullPath;
        }

        private static string GetSubmissionsFolderPath()
        {
            string folder = SubmissionDemoConfiguration.Configuration.SubmissionsPath.Path;
            string subfolder = SUBMISSIONS_FOLDER;
            string submissionsPath = Path.Combine(folder, subfolder);
            if (Directory.Exists(submissionsPath) == false) Directory.CreateDirectory(submissionsPath);
            return submissionsPath;
        }


        private static SubmissionRecord GetSubmission(string filePath)
        {
            if (string.IsNullOrWhiteSpace(filePath) == true) return null;
            if (File.Exists(filePath) == false) return null;

            using (FileStream fs = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                XmlSerializer xs = new XmlSerializer(typeof(SubmissionRecord));
                try
                {
                    return (SubmissionRecord)xs.Deserialize(fs);
                }
                catch (Exception ex)
                {
                    Trace.WriteLine(string.Format("Skipping submission file {0}, exception: {1}", filePath, ex.ToString()));
                    return null;
                }

            }
        }

        public static string Data { get { return GetSubmissionsFolderPath(); } }

        public static List<SubmissionRecord> LoadSubmissions(bool onlyActive)
        {
            List<SubmissionRecord> submissions = new List<SubmissionRecord>();
            var query = from fi in Directory.GetFiles(GetSubmissionsFolderPath(), INDEX_FILE, SearchOption.AllDirectories).AsParallel() select fi;

            Parallel.ForEach(query, file =>
            {
                SubmissionRecord record = GetSubmission(file);
                if (record != null)
                {
                    if (((onlyActive == true) && record.Status != SubmissionStatus.finished) || onlyActive == false)
                    {
                        lock (lockObj)
                        {
                            submissions.Add(record);
                        }
                    }
                }
            });
            return submissions.OrderBy(s => s.DateCreated).ToList<SubmissionRecord>();
        }

        public static void AddSubmission(SubmissionRecord record)
        {
            using (FileStream fs = new FileStream(GetSubmissionFilePath(record.InternalID, INDEX_FILE), FileMode.Create, FileAccess.Write, FileShare.Read))
            {
                XmlSerializer xs = new XmlSerializer(typeof(SubmissionRecord));
                xs.Serialize(fs, record);
            }
        }

        public static void AddSubmissionData(string internalID, byte[] data, string dataID)
        {
            if (data == null) return;
            using (FileStream fs = new FileStream(GetSubmissionFilePath(internalID, dataID), FileMode.Create, FileAccess.Write, FileShare.Read))
            {
                fs.Write(data, 0, data.Length);
            }
        }

        public static void SetStatus(string internalID, SubmissionStatus status)
        {
            List<SubmissionRecord> submissions = LoadSubmissions(false);
            SubmissionRecord record = submissions.Find(s => s.InternalID == internalID);
            if (record != null) record.Status = status;

            using (FileStream fs = new FileStream(GetSubmissionFilePath(record.InternalID, INDEX_FILE), FileMode.Create, FileAccess.Write, FileShare.Read))
            {
                XmlSerializer xs = new XmlSerializer(typeof(SubmissionRecord));
                xs.Serialize(fs, record);
            }
        }

        public static List<SubmissionDetails> GetSubmissionDetails(SubmissionRecord r)
        {
            List<SubmissionDetails> d = new List<SubmissionDetails>();
            if (r.InternalID != null)
            {
                var query = from fi in Directory.GetFiles(Path.Combine(GetSubmissionsFolderPath(), r.InternalID), "*.*", SearchOption.TopDirectoryOnly).AsParallel() select fi;

                Parallel.ForEach(query, file =>
                {


                    if (Path.GetFileName(file).ToLower() != INDEX_FILE.ToLower())
                    {
                        lock (lockObj)
                        {
                            d.Add(new SubmissionDetails(file));
                        }
                    }

                });
                return d.OrderBy(dd => dd.ModificationTime).ToList<SubmissionDetails>();
            }
            else
            {
                return d;
            }
        }

        public static string GetSubmissionDetailContent(SubmissionDetails d)
        {
            try
            {
                if (d != null && File.Exists(d.FilePath))
                {
                    using (FileStream fs = new FileStream(d.FilePath, FileMode.Open, FileAccess.Read, FileShare.ReadWrite))
                    {
                        using (StreamReader sr = new StreamReader(fs))
                        {
                            string s = sr.ReadToEnd();

                            return s;
                        }
                    }
                }
                else
                {
                    return null;
                }
            }
            catch (Exception ex)
            {
                Trace.Write(string.Format("Exception when loading submission detail content: {0}", ex.ToString()));
                return null;
            }
        }
    }
}
