﻿using System;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Security.Cryptography.Pkcs;
using System.Text;
using System.Xml;
using CSSZSubmissionDemo.Settings;

namespace CSSZSubmissionDemo.Channels
{
    class ISDSChannel : ChannelBase
    {
        protected override string InternalDisplayName()
        {
            return "ISDS";
        }

        public override void SendSubmission(Submissions.SubmissionBase submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Encoding submission"));
            byte[] data = GetMinimalMessage(ref submission);
            OnSubmissionData(new SubmissionDataEventArgs(submission.InternalID, ref data, "submit"));

            Trace.WriteLine(string.Format("Initialize service"));
            WSOps.dmOperationsWebService svcOps = new WSOps.dmOperationsWebService();
            svcOps.UserAgent = "CSSZSubmissionDemo(.NET)";
            svcOps.AllowAutoRedirect = true;
            svcOps.PreAuthenticate = true;
            svcOps.Proxy = WebRequest.DefaultWebProxy;
            svcOps.Timeout = Channels.ChannelBase.TIMEOUT * 1000;

            //TODO: ISDS credentials and urls combinations
            svcOps.Url = String.Format("{0}{1}", ((Configuration.ISDSChannelElement)settings).URIBase.User, ((Configuration.ISDSChannelElement)settings).URISuffix.Operations);
            //X509Certificate2 c = Helpers.X509Certificate2Helpers.Load(((Configuration.ISDSChannelElement)settings).Sender);
            //svcOps.ClientCertificates.Add(c);
            svcOps.Credentials = new NetworkCredential(((Configuration.ISDSChannelElement)settings).Sender.Login, ((Configuration.ISDSChannelElement)settings).Sender.Password);

            DateTime d = DateTime.Now;
            WSOps.tMessageCreateOutput tmco = svcOps.CreateMessage(new WSOps.tMessageCreateInput()
            {
                dmEnvelope = new WSOps.tMessageEnvelopeSub()
                {
                    dbIDRecipient = ((Configuration.ISDSChannelElement)settings).Recipient,
                    dmAnnotation = String.Format("Podani {1} {0:yyyyMMddHHmmss}", d, submission.ClassName),
                    dmLegalTitleLaw = "",
                    dmLegalTitlePar = "",
                    dmLegalTitlePoint = "",
                    dmLegalTitleSect = "",
                    dmLegalTitleYear = "",
                    dmRecipientOrgUnit = "",
                    dmRecipientOrgUnitNum = "",
                    dmRecipientRefNumber = "", //cislo jednaci znacka prijemce
                    dmRecipientIdent = "", //spisova znacka prijemce
                    dmSenderOrgUnit = "", //OU podavajiciho
                    dmSenderOrgUnitNum = "", //OU podavajiciho
                    dmSenderRefNumber = "CJ" + new System.Random().Next(1000, 9999).ToString(), //cislo jednaci podavajiciho
                    dmSenderIdent = "SZ" + new System.Random().Next(1000, 9999).ToString(), //spisova znacka podavajiciho
                    dmToHands = ""
                },
                dmFiles = new WSOps.tFilesArrayDmFile[] {
                    new WSOps.tFilesArrayDmFile() {
                        dmFileMetaType = WSOps.tFilesArrayDmFileDmFileMetaType.main,
                        dmMimeType = "application/xml",
                        dmFileDescr = String.Format("Podani-{0:yyyyMMddHHmmss}.xml",d),
                        Item = data
                    }
                }
            });

            Trace.WriteLine(string.Format("Decoding response"));
            byte[] rsp = Encoding.UTF8.GetBytes(tmco.dmID);
            DAL.SubmissionStatus s;
            AckInfo ack = this.DecodeAck(ref rsp, submission, out s);
            OnSubmissionCreated(new SubmissionCreatedEventArgs(new DAL.SubmissionRecord(this.Ack, this.DisplayName)));

            DownloadSentMessage(ref svcOps, tmco.dmID, submission.InternalID);
        }

        public override void PollForResponse(Settings.SubmissionInfo submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Initialize service"));
            WSOps.dmOperationsWebService svcOps = new WSOps.dmOperationsWebService();
            svcOps.UserAgent = "CSSZSubmissionDemo(.NET)";
            svcOps.AllowAutoRedirect = true;
            svcOps.PreAuthenticate = true;
            svcOps.Proxy = WebRequest.DefaultWebProxy;
            svcOps.Timeout = Channels.ChannelBase.TIMEOUT * 1000;

            //TODO: ISDS credentials and urls combinations
            svcOps.Url = String.Format("{0}{1}", ((Configuration.ISDSChannelElement)settings).URIBase.User, ((Configuration.ISDSChannelElement)settings).URISuffix.Operations);
            //X509Certificate2 c = Helpers.X509Certificate2Helpers.Load(((Configuration.ISDSChannelElement)settings).Sender);
            //svcOps.ClientCertificates.Add(c);
            svcOps.Credentials = new NetworkCredential(((Configuration.ISDSChannelElement)settings).Sender.Login, ((Configuration.ISDSChannelElement)settings).Sender.Password);

            WSInfo.dmInfoWebService svcInfo = new WSInfo.dmInfoWebService();
            svcInfo.UserAgent = "CSSZSubmissionDemo(.NET)";
            svcInfo.AllowAutoRedirect = true;
            svcInfo.PreAuthenticate = true;
            svcInfo.Proxy = WebRequest.DefaultWebProxy;
            svcInfo.Timeout = Channels.ChannelBase.TIMEOUT * 1000;

            //TODO: ISDS credentials and urls combinations
            svcInfo.Url = String.Format("{0}{1}", ((Configuration.ISDSChannelElement)settings).URIBase.User, ((Configuration.ISDSChannelElement)settings).URISuffix.Operations);
            //X509Certificate2 c = Helpers.X509Certificate2Helpers.Load(((Configuration.ISDSChannelElement)settings).Sender);
            //svcInfo.ClientCertificates.Add(c);
            svcInfo.Credentials = new NetworkCredential(((Configuration.ISDSChannelElement)settings).Sender.Login, ((Configuration.ISDSChannelElement)settings).Sender.Password);

            Trace.WriteLine(string.Format("List received messages"));
            WSInfo.tListOfMessOutput tlmo = svcInfo.GetListOfReceivedMessages(new WSInfo.tListOfFReceivedInput() { dmFromTime = DateTime.Now.Date.AddDays(-5), dmToTime = DateTime.Now.Date.AddDays(1) });

            if (tlmo != null && tlmo.dmRecords != null && tlmo.dmRecords.dmRecord != null)
            {
                foreach (var dm in tlmo.dmRecords.dmRecord)
                {
                    Trace.WriteLine(string.Format("Iterate download without attachments"));
                    if (dm.dmAnnotation.Contains(submission.ID) == true)
                    {
                        Trace.WriteLine(string.Format("Download signed message response"));
                        WSOps.tSignedMessDownOutput tsmo = svcOps.SignedMessageDownload(new WSOps.tIDMessInput() { dmID = dm.dmID });
                        byte[] rsp = tsmo.dmSignature;
                        OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref rsp, "response.zfo"));

                        try
                        {
                            Trace.WriteLine(string.Format("Decode signed message response"));
                            SignedCms scms = new SignedCms();
                            scms.Decode(rsp);
                            byte[] rspd = scms.ContentInfo.Content;

                            if (rspd != null && rspd.Length > 0)
                                OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref rspd, "response.decoded.zfo"));
                        }
                        catch (Exception ex)
                        {
                            Trace.WriteLine(string.Format("Exception while decoding ISDS message: {0}", ex.ToString()));
                        }

                        Trace.WriteLine(string.Format("Download unsigned message response"));
                        WSOps.tMessDownOutput tmo = svcOps.MessageDownload(new WSOps.tIDMessInput() { dmID = dm.dmID });
                        if (tmo.dmReturnedMessage.dmDm.dmFiles.Count() == 1
                            && tmo.dmReturnedMessage.dmDm.dmFiles[0].dmFileMetaType == WSOps.tFilesArrayDmFileDmFileMetaType.main
                            && tmo.dmReturnedMessage.dmDm.dmFiles[0].dmMimeType == "application/xml")
                        {
                            Trace.WriteLine(string.Format("Parse XML from response"));
                            WSOps.tFilesArrayDmFile f = tmo.dmReturnedMessage.dmDm.dmFiles[0];
                            rsp = (byte[])f.Item;
                            OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref rsp, String.Format("response.xml.{0}", f.dmFileDescr.Substring(f.dmFileDescr.IndexOf(submission.ClassName, StringComparison.CurrentCultureIgnoreCase)))));

                            DAL.SubmissionStatus s;
                            AckInfo ack = this.DecodeResp(ref rsp, submission, out s);
                            OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, s));
                        }
                    }
                }
            }

            Trace.WriteLine(string.Format("Download sent message"));
            DownloadSentMessage(ref svcOps, submission.ID, submission.internalID);
        }

        public override void CompleteTransaction(Settings.SubmissionInfo submission)
        {
            //ISDS nevyzaduje zadne ukonceni transakce
            DAL.SubmissionStatus status = DAL.SubmissionStatus.finished;
            OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, status));
            return;
        }

        protected override byte[] GetMinimalMessage(ref Submissions.SubmissionBase submission)
        {
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(Encoding.UTF8.GetString(submission.Encode()));

            //SenderDetails pro ISDS nejsou pouzivany
            XmlNode nod = doc.SelectSingleNode("//*[local-name()='SenderDetails']");
            if (nod != null) nod.ParentNode.RemoveChild(nod);

            //GatewayAdditions pro ISDS nejsou pouzivany
            nod = doc.SelectSingleNode("//*[local-name()='GatewayAdditions']");
            if (nod != null) nod.ParentNode.RemoveChild(nod);

            if (submission.ClassName == "CSSZ_HPN")
            {
                //HPN nevyzaduje Keys, takze ani GovTalkDetails 
                nod = doc.SelectSingleNode("//*[local-name()='GovTalkDetails']");
                if (nod != null) nod.ParentNode.RemoveChild(nod);
            }

            return Encoding.UTF8.GetBytes(doc.OuterXml.ToString());
        }

        private void DownloadSentMessage(ref WSOps.dmOperationsWebService svc, string dmID, string internalID)
        {
            WSOps.tSignedMessDownOutput tsmo = svc.SignedSentMessageDownload(new WSOps.tIDMessInput() { dmID = dmID });
            if (tsmo.dmStatus.dmStatusCode == "0000")
            {
                byte[] snt = tsmo.dmSignature;
                OnSubmissionData(new SubmissionDataEventArgs(internalID, ref snt, "sent.zfo"));
            }
        }

        private Settings.AckInfo InternalDecodeAck(ref byte[] response, string className, string internalID, string VS, out DAL.SubmissionStatus status)
        {
            string id = Encoding.UTF8.GetString(response);
            status = DAL.SubmissionStatus.created;

            ack = new Settings.AckInfo() { ID = id, internalID = internalID, Class = className, status = status.ToString(), VS = VS };
            return ack;
        }

        public override AckInfo DecodeAck(ref byte[] response, SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecodeAck(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public override AckInfo DecodeAck(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecodeAck(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }

        private Settings.AckInfo InternalDecodeResp(ref byte[] response, string className, string internalID, string VS, out DAL.SubmissionStatus status)
        {
            //vsechny odpovedi pres ISDS jsou ve formatu GovTalk
            Submissions.GovTalkResponse r = new Submissions.GovTalkResponse();
            r.Initialize(ref response, null, null, null);
            AckInfo a = new AckInfo();
            //GovTalk parser
            byte[] data = r.Decode(ref response, out a);

            //pokud jsou v odpovedi obsazena data
            if (data != null) OnSubmissionResponse(new SubmissionDataEventArgs(internalID, ref data, "response.decoded"));

            //Odpoved pres ISDS znamena, ze uz neni treba dalsi polling
            status = DAL.SubmissionStatus.sent;

            //set additional properties
            ack = a;
            ack.internalID = internalID;
            ack.Class = className;
            ack.VS = VS;
            ack.status = status.ToString();
            return ack;
        }

        public override AckInfo DecodeResp(ref byte[] response, SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecodeResp(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public override AckInfo DecodeResp(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecodeResp(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }
    }
}
