﻿using System.Diagnostics;
using System.IO;
using System.Net;
using System.Text;
using System.Xml;
using CSSZSubmissionDemo.Helpers.Extensions;
using CSSZSubmissionDemo.Settings;

namespace CSSZSubmissionDemo.Channels
{
    class VREPChannel : ChannelBase
    {
        protected override string InternalDisplayName()
        {
            return "VREP";
        }

        public override void SendSubmission(Submissions.SubmissionBase submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Encoding submission"));
            byte[] data = GetMinimalMessage(ref submission);
            OnSubmissionData(new SubmissionDataEventArgs(submission.InternalID, ref data, "submit"));

            byte[] resp;
            string status;
            InternalExecute(((Configuration.VREPChannelElement)settings).URI.Submission, data, out resp, out status);
            OnSubmissionData(new SubmissionDataEventArgs(submission.InternalID, ref resp, "ack"));

            Trace.WriteLine(string.Format("Decoding ack"));
            DAL.SubmissionStatus s;
            AckInfo ack = this.DecodeAck(ref resp, submission, out s);
            OnSubmissionCreated(new SubmissionCreatedEventArgs(new DAL.SubmissionRecord(this.Ack, this.DisplayName)));
        }

        public override void PollForResponse(Settings.SubmissionInfo submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Encoding poll"));
            byte[] data = PrepareRequestData(submission.ClassName, "submit", "poll", submission.ID, submission.VS);
            OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref data, "poll"));

            byte[] resp;
            string status;
            InternalExecute(((Configuration.VREPChannelElement)settings).URI.Poll, data, out resp, out status);
            OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref resp, "resp"));

            Trace.WriteLine(string.Format("Decoding response"));
            DAL.SubmissionStatus s;
            AckInfo ack = this.DecodeAck(ref resp, submission, out s);
            OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, s));
        }

        public override void CompleteTransaction(Settings.SubmissionInfo submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Encoding delete"));
            byte[] data = PrepareRequestData(submission.ClassName, "delete", "request", submission.ID, submission.VS);
            OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref data, "poll"));

            byte[] resp;
            string status;
            InternalExecute(((Configuration.VREPChannelElement)settings).URI.Submission, data, out resp, out status);
            OnSubmissionData(new SubmissionDataEventArgs(submission.internalID, ref resp, "close"));

            Trace.WriteLine(string.Format("Decoding response"));
            DAL.SubmissionStatus s;
            AckInfo ack = this.DecodeResp(ref resp, submission, out s);
            OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, s));
        }

        private byte[] PrepareRequestData(string className, string function, string qualifier, string correlationId, string VS)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine("Load GovTalk Template");
            XmlDocument doc = new XmlDocument();
            XmlNode nod = null;
            doc.LoadFromTemplate("GovTalkMessageTemplate.xml");
            Trace.WriteLine("Set XML valuess");
            doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='Header']/*[local-name()='MessageDetails']/*[local-name()='Class']").InnerText = className;
            doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='Header']/*[local-name()='MessageDetails']/*[local-name()='Qualifier']").InnerText = qualifier;
            doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='Header']/*[local-name()='MessageDetails']/*[local-name()='Function']").InnerText = function;
            doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='Header']/*[local-name()='MessageDetails']/*[local-name()='CorrelationID']").InnerText = correlationId;
            doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='GovTalkDetails']/*[local-name()='Keys']/*[local-name()='Key' and @Type='vars']").InnerText = VS;

            if (string.IsNullOrWhiteSpace(((Configuration.VREPChannelElement)settings).Email) == true)
            {
                nod = doc.SelectSingleNode("//*[local-name()='SenderDetails']/*[local-name()='EmailAddress']");
                if (nod != null) nod.ParentNode.RemoveChild(nod);
            }
            else
            {
                doc.SelectSingleNode("//*[local-name()='SenderDetails']/*[local-name()='EmailAddress']").InnerText = ((Configuration.VREPChannelElement)settings).Email;
            }

            //SenderDetails pro VREP nejsou pouzivany
            nod = doc.SelectSingleNode("//*[local-name()='SenderDetails']");
            if (nod != null) nod.ParentNode.RemoveChild(nod);

            //GovTalkDetails jsou pro VREP pouzivany
            //GatewayAdditions jsou pro VREP pouzivany

            //HPN nevyzaduje Keys 
            if (className == "CSSZ_HPN")
            {
                nod = doc.SelectSingleNode("//*[local-name()='Keys']");
                if (nod != null) nod.ParentNode.RemoveChild(nod);
            }

            return Encoding.UTF8.GetBytes(doc.OuterXml);
        }

        private void InternalExecute(string URI, byte[] data, out byte[] resp, out string status)
        {
            Trace.WriteLine(string.Format("Preparing web request"));
            HttpWebRequest rq = (HttpWebRequest)WebRequest.Create(URI);
            rq.UserAgent = "CSSZSubmissionDemo(.NET)";
            rq.KeepAlive = false;
            rq.ProtocolVersion = HttpVersion.Version10;
            rq.Method = "POST";
            rq.Proxy = WebRequest.DefaultWebProxy;
            rq.Timeout = Channels.ChannelBase.TIMEOUT * 1000;

            rq.ContentType = "text/xml";
            rq.ContentLength = data.Length;
            //rq.TransferEncoding = "";

            Stream rqs = rq.GetRequestStream();

            rqs.Write(data, 0, data.Length);
            rqs.Close();

            Trace.WriteLine(string.Format("Sending request"));
            HttpWebResponse rsp = (HttpWebResponse)rq.GetResponse();
            Trace.WriteLine(string.Format("Reading response"));
            Stream s = rsp.GetResponseStream();
            StreamReader sr = new StreamReader(s);

            resp = Encoding.UTF8.GetBytes(sr.ReadToEnd());
            status = rsp.StatusCode.ToString();
        }

        private AckInfo InternalDecode(ref byte[] response, string className, string internalID, string VS, out DAL.SubmissionStatus status)
        {
            //vsechny odpovedi (ack, response, i error jsou ve formatu GovTalk
            Submissions.GovTalkResponse r = new Submissions.GovTalkResponse();
            r.Initialize(ref response, null, null, null);
            AckInfo a = new AckInfo();
            //GovTalk parser
            byte[] data = r.Decode(ref response, out a);

            status = DAL.SubmissionStatus.created;
            if (a.function.ToLower() == "submit" && (a.qualifier.ToLower() == "response" || (a.qualifier.ToLower() == "error" && string.IsNullOrWhiteSpace(a.ID) == false)))
            {
                status = DAL.SubmissionStatus.sent;
                //pokud jsou v odpovedi obsazena data
                if (data != null) OnSubmissionResponse(new SubmissionDataEventArgs(internalID, ref data, "response.decoded"));
            }
            if (a.function.ToLower() == "delete" && a.qualifier.ToLower() == "response")
            {
                status = DAL.SubmissionStatus.finished;
            }

            //set additional properties
            ack = a;
            ack.internalID = internalID;
            ack.Class = className;
            ack.VS = VS;
            ack.status = status.ToString();
            return ack;
        }

        protected override byte[] GetMinimalMessage(ref Submissions.SubmissionBase submission)
        {
            XmlDocument doc = new XmlDocument();
            doc.LoadXml(Encoding.UTF8.GetString(submission.Encode()));

            //SenderDetails pro VREP nejsou pouzivany
            XmlNode nod = doc.SelectSingleNode("//*[local-name()='SenderDetails']");
            if (nod != null) nod.ParentNode.RemoveChild(nod);
            //GovTalkDetails jsou pro VREP pouzivany
            //GatewayAdditions jsou pro VREP pouzivany

            //HPN nevyzaduje Keys/VS 
            if (submission.ClassName == "CSSZ_HPN")
            {
                nod = doc.SelectSingleNode("//*[local-name()='Keys']");
                if (nod != null) nod.ParentNode.RemoveChild(nod);
            }

            return Encoding.UTF8.GetBytes(doc.OuterXml.ToString());
        }

        public override AckInfo DecodeAck(ref byte[] response, SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public override AckInfo DecodeAck(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }

        public override AckInfo DecodeResp(ref byte[] response, SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public override AckInfo DecodeResp(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }
    }
}
