﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading;
using System.Windows.Forms;
using System.Xml;
using CSSZSubmissionDemo.Configuration;
using CSSZSubmissionDemo.Helpers.Extensions;

namespace CSSZSubmissionDemo
{
    public partial class frmMain : Form
    {
        [DllImport("kernel32.dll")]
        public static extern Boolean AllocConsole();

        [DllImport("kernel32.dll")]
        public static extern Boolean FreeConsole();

        private bool dirty = false;
        private bool showOnlyActiveSubmissions = true;
        private bool consoleRunning = false;
        private object lockObj = new object();
        private System.Timers.Timer tmr;

        List<ChannelElement> channels = new List<ChannelElement>();
        List<SubmissionElement> submissions = new List<SubmissionElement>();
        List<IdentifierElement> identifiers = new List<IdentifierElement>();
        List<string> formats = new List<string>() { "GovTalk", "Plain XML" };
        Dictionary<string, string> samples = new Dictionary<string, string>();

        public frmMain()
        {
            InitializeComponent();
            showOnlyActiveSubmissions = this.activeToolStripMenuItem.Checked;

            dgSubmissions.AutoGenerateColumns = true;
            dgSubmissions.DataSource = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions);

            foreach (ISDSChannelElement ch in SubmissionDemoConfiguration.Configuration.ISDSChannels)
            {
                channels.Add(ch);
            }

            foreach (VREPChannelElement ch in SubmissionDemoConfiguration.Configuration.VREPChannels)
            {
                channels.Add(ch);
            }

            foreach (VREPWSChannelElement ch in SubmissionDemoConfiguration.Configuration.VREPWSChannels)
            {
                channels.Add(ch);
            }

            foreach (SubmissionElement s in SubmissionDemoConfiguration.Configuration.SubmissionTypes)
            {
                submissions.Add(s);
            }

            foreach (IdentifierElement i in SubmissionDemoConfiguration.Configuration.Identifiers)
            {
                identifiers.Add(i);
            }

            samples.Add("", null);
            try
            {
                if (Directory.Exists(SubmissionDemoConfiguration.Configuration.SamplesPath.Path))
                {
                    foreach (string fl in Directory.GetFiles(SubmissionDemoConfiguration.Configuration.SamplesPath.Path, "*.xml"))
                    {
                        samples.Add(Path.GetFileName(fl), fl);
                    }
                }

            }
            catch (Exception)
            {
                //intentionally left blank
            }

            foreach (ChannelElement ch in channels)
            {
                ToolStripMenuItem mip = new ToolStripMenuItem(ch.DisplayName);
                mip.Click += new EventHandler(ChannelPoll_Click);
                mip.Tag = ch;
                this.pollByChannelToolStripMenuItem.DropDownItems.Add(mip);

                ToolStripMenuItem mic = new ToolStripMenuItem(ch.DisplayName);
                mic.Click += new EventHandler(ChannelClose_Click);
                mip.Tag = ch;
                this.closeByChannelToolStripMenuItem.DropDownItems.Add(mic);
            }

            InitThreadPool();

            tmr = new System.Timers.Timer();
            tmr.AutoReset = true;
            tmr.Interval = 10000;
            tmr.Elapsed += new System.Timers.ElapsedEventHandler(tmr_Elapsed);
            tmr.Start();
        }

        private static void InitThreadPool()
        {
            int mxwt; int mxcpt; int miwt; int micpt;
            ThreadPool.GetMinThreads(out miwt, out micpt);
            ThreadPool.GetMaxThreads(out mxwt, out mxcpt);
            Trace.Write(string.Format("{0} {1} {2} {3}", miwt, mxwt, micpt, mxcpt));
            ThreadPool.SetMinThreads(50, 50);
            ThreadPool.GetMinThreads(out miwt, out micpt);
            ThreadPool.GetMaxThreads(out mxwt, out mxcpt);
            Trace.Write(string.Format("{0} {1} {2} {3}", miwt, mxwt, micpt, mxcpt));
        }

        void tmr_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            if (dirty)
            {
                dirty = false;
                Reload();
            }
        }

        private void Dirty()
        {
            lock (lockObj)
            {
                if (!dirty)
                {
                    dirty = true;
                }
            }
        }

        void ch_SubmissionCreated(object sender, Channels.SubmissionCreatedEventArgs e)
        {
            Helpers.Worker w = new Helpers.Worker();
            w.SaveSubmission(e.Record);
            Dirty();
        }

        void ch_SubmissionStatus(object sender, Channels.SubmissionStatusEventArgs e)
        {
            Helpers.Worker w = new Helpers.Worker();
            w.UpdateSubmissionStatus(e.InternalID, e.Status);
            Dirty();
        }

        void ch_SubmissionResponse(object sender, Channels.SubmissionDataEventArgs e)
        {
            byte[] data = e.Data;
            if (data != null && data.Length > 0)
            {
                Helpers.Worker w = new Helpers.Worker();
                w.AddSubmissionData(e.InternalID, ref data, "response.decoded");
            }
            Dirty();
        }

        void w_Finished(object sender, EventArgs e)
        {
            Dirty();
        }

        void ch_SubmissionData(object sender, Channels.SubmissionDataEventArgs e)
        {
            Helpers.Worker w = new Helpers.Worker();
            byte[] data = e.Data;
            w.AddSubmissionData(e.InternalID, ref data, e.DataId);
            Dirty();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Environment.Exit(0);
        }

        private void submissionsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Process.Start("explorer.exe", DAL.Connection.Data);
        }

        private void refreshToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Reload();
        }

        private void newToolStripMenuItem_Click(object sender, EventArgs e)
        {
            frmSubmission frmS = new frmSubmission(submissions, channels, formats, identifiers, samples);
            DialogResult dr = frmS.ShowDialog(this);
            if (dr == DialogResult.OK || dr == DialogResult.Yes)
            {
                Submissions.SubmissionBase s = frmS.submission;
                Channels.ChannelBase ch = Channels.ChannelBase.CreateChannel(frmS.channel);
                ch.SubmissionCreated += new Channels.SubmissionCreatedEventHandler(ch_SubmissionCreated);
                ch.SubmissionData += new Channels.SubmissionDataEventHandler(ch_SubmissionData);
                ch.SubmissionStatus += new Channels.SubmissionStatusEventHandler(ch_SubmissionStatus);

                Helpers.Worker w = new Helpers.Worker() { submission = s, channel = ch };
                w.Sent += new EventHandler(w_Finished);
                ThreadPool.QueueUserWorkItem(new WaitCallback(w.SendSubmission));
            }
        }

        private void pollToolStripMenuItem_Click(object sender, EventArgs e)
        {
            PollTransactions(null);
        }

        void ChannelPoll_Click(object sender, EventArgs e)
        {
            if (sender != null)
            {
                ToolStripMenuItem mi = (ToolStripMenuItem)sender;
                if (mi != null && mi.Tag != null)
                {
                    ChannelElement ch = (ChannelElement)mi.Tag;
                    PollTransactions(ch.DisplayName);
                }
            }

        }

        private void PollTransactions(string channelDisplayName)
        {
            IEnumerable<DAL.SubmissionRecord> query;
            if (string.IsNullOrWhiteSpace(channelDisplayName))
            {
                query = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => s.Status != DAL.SubmissionStatus.finished && string.IsNullOrWhiteSpace(s.ChannelDisplayName) == false && string.IsNullOrWhiteSpace(s.InternalID) == false);
            }
            else
            {
                query = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => s.Status != DAL.SubmissionStatus.finished && string.IsNullOrWhiteSpace(s.ChannelDisplayName) == false && string.IsNullOrWhiteSpace(s.InternalID) == false && s.ChannelDisplayName == channelDisplayName);
            }
            foreach (var submission in query)
            {
                Channels.ChannelBase ch = Channels.ChannelBase.CreateChannel(channels.FirstOrDefault(chi => chi.DisplayName == submission.ChannelDisplayName));
                ch.SubmissionData += new Channels.SubmissionDataEventHandler(ch_SubmissionData);
                ch.SubmissionStatus += new Channels.SubmissionStatusEventHandler(ch_SubmissionStatus);
                ch.SubmissionResponse += new Channels.SubmissionDataEventHandler(ch_SubmissionResponse);
                Settings.SubmissionInfo s = new Settings.SubmissionInfo() { ID = submission.ID, ClassName = submission.Class, internalID = submission.InternalID, VS = submission.VS };
                Helpers.Worker w = new Helpers.Worker() { submissionInfo = s, channel = ch };

                ThreadPool.QueueUserWorkItem(new WaitCallback(w.PollResponse));
            }
        }

        private void closeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            CloseTransactions(null);
        }

        void ChannelClose_Click(object sender, EventArgs e)
        {
            if (sender != null)
            {
                ToolStripMenuItem mi = (ToolStripMenuItem)sender;
                if (mi != null && mi.Tag != null)
                {
                    ChannelElement ch = (ChannelElement)mi.Tag;
                    CloseTransactions(ch.DisplayName);
                }
            }
        }

        private void CloseTransactions(string channelDisplayName)
        {
            IEnumerable<DAL.SubmissionRecord> query;
            if (string.IsNullOrWhiteSpace(channelDisplayName))
            {
                query = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => s.Status == DAL.SubmissionStatus.sent && string.IsNullOrWhiteSpace(s.ChannelDisplayName) == false && string.IsNullOrWhiteSpace(s.InternalID) == false && string.IsNullOrWhiteSpace(s.ID) == false);
            }
            else
            {
                query = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => s.Status == DAL.SubmissionStatus.sent && string.IsNullOrWhiteSpace(s.ChannelDisplayName) == false && string.IsNullOrWhiteSpace(s.InternalID) == false && string.IsNullOrWhiteSpace(s.ID) == false && s.ChannelDisplayName == channelDisplayName);
            }

            foreach (var submission in query)
            {
                Channels.ChannelBase ch = Channels.ChannelBase.CreateChannel(channels.FirstOrDefault(chi => chi.DisplayName == submission.ChannelDisplayName));
                ch.SubmissionData += new Channels.SubmissionDataEventHandler(ch_SubmissionData);
                ch.SubmissionStatus += new Channels.SubmissionStatusEventHandler(ch_SubmissionStatus);
                Settings.SubmissionInfo s = new Settings.SubmissionInfo() { ID = submission.ID, ClassName = submission.Class, internalID = submission.InternalID, VS = submission.VS };
                Helpers.Worker w = new Helpers.Worker() { submissionInfo = s, channel = ch };

                ThreadPool.QueueUserWorkItem(new WaitCallback(w.CloseTransaction));
            }
        }

        private void closeForceToolStripMenuItem_Click(object sender, EventArgs e)
        {
            foreach (var submission in DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => (s.Status == DAL.SubmissionStatus.sent || s.Status == DAL.SubmissionStatus.created) && string.IsNullOrWhiteSpace(s.InternalID) == false))
            {
                Channels.ChannelBase ch = Channels.ChannelBase.CreateChannel(channels.FirstOrDefault(chi => chi.DisplayName == submission.ChannelDisplayName));
                Settings.SubmissionInfo s = new Settings.SubmissionInfo() { ID = submission.ID, ClassName = submission.Class, internalID = submission.InternalID, VS = submission.VS };
                Helpers.Worker w = new Helpers.Worker() { submissionInfo = s, channel = ch };
                w.UpdateSubmissionStatus(s.internalID, DAL.SubmissionStatus.finished);
            }
        }

        private void closeInvalidToolStripMenuItem_Click(object sender, EventArgs e)
        {
            foreach (var submission in DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions).Where(s => (s.Status == DAL.SubmissionStatus.sent || s.Status == DAL.SubmissionStatus.created) && (string.IsNullOrWhiteSpace(s.ChannelDisplayName) == true || string.IsNullOrWhiteSpace(s.ID) == true) && string.IsNullOrWhiteSpace(s.InternalID) == false))
            {
                Channels.ChannelBase ch = Channels.ChannelBase.CreateChannel(channels.FirstOrDefault(chi => chi.DisplayName == submission.ChannelDisplayName));
                Settings.SubmissionInfo s = new Settings.SubmissionInfo() { ID = submission.ID, ClassName = submission.Class, internalID = submission.InternalID, VS = submission.VS };
                Helpers.Worker w = new Helpers.Worker() { submissionInfo = s, channel = ch };
                w.UpdateSubmissionStatus(s.internalID, DAL.SubmissionStatus.finished);
            }
        }

        private void activeToolStripMenuItem_CheckedChanged(object sender, EventArgs e)
        {
            showOnlyActiveSubmissions = this.activeToolStripMenuItem.Checked;
            Reload();
        }

        private void traceToolStripMenuItem_CheckedChanged(object sender, EventArgs e)
        {
            if (this.traceToolStripMenuItem.Checked && !consoleRunning)
            {
                AllocConsole();
                Helpers.ConsoleListener l = new Helpers.ConsoleListener() { Name = "SDCTL" };

                Trace.Listeners.Add(l);

                Console.BufferWidth = 200;
                Console.BufferHeight = 1000;
                consoleRunning = true;
            }
            else if (!this.traceToolStripMenuItem.Checked && consoleRunning)
            {
                FreeConsole();
                consoleRunning = false;
            }
            else
            {
                //intentionally left blank
            }
        }

        private void Reload()
        {
            if (dgSubmissions.InvokeRequired)
            {
                this.Invoke(new MethodInvoker(Reload));
            }
            else
            {
                dgSubmissions.DataSource = null;
                dgSubmissions.Rows.Clear();
                dgSubmissions.Columns.Clear();
                dgSubmissions.AutoGenerateColumns = true;
                dgSubmissions.DataSource = DAL.Connection.LoadSubmissions(showOnlyActiveSubmissions);
                dgSubmissions.AutoResizeColumns(DataGridViewAutoSizeColumnsMode.AllCells);
            }
        }

        private void dgSubmissions_SelectionChanged(object sender, EventArgs e)
        {
            dgDetails.DataSource = null;
            dgDetails.Rows.Clear();
            dgDetails.Columns.Clear();
            dgDetails.AutoGenerateColumns = true;

            if (dgSubmissions.SelectedRows.Count == 1 && dgSubmissions.SelectedRows[0] != null && dgSubmissions.SelectedRows[0].DataBoundItem != null)
            {
                DAL.SubmissionRecord r = (DAL.SubmissionRecord)dgSubmissions.SelectedRows[0].DataBoundItem;
                dgDetails.DataSource = DAL.Connection.GetSubmissionDetails(r);
                dgDetails.AutoResizeColumns(DataGridViewAutoSizeColumnsMode.AllCells);
            }
        }

        private void dgDetails_SelectionChanged(object sender, EventArgs e)
        {
            if (dgDetails.SelectedRows.Count == 1 && dgDetails.SelectedRows[0] != null && dgDetails.SelectedRows[0].DataBoundItem != null)
            {
                DAL.SubmissionDetails d = (DAL.SubmissionDetails)dgDetails.SelectedRows[0].DataBoundItem;

                string s = DAL.Connection.GetSubmissionDetailContent(d);
                if (!string.IsNullOrWhiteSpace(s))
                {
                    txtData.Text = s;
                    XmlDocument doc = new XmlDocument();
                    txtFormated.Text = doc.FormattedXml(s);
                }
                else
                {
                    txtData.Text = "";
                    txtFormated.Text = "";
                }
            }
        }
    }
}
