﻿using System;
using System.Diagnostics;
using System.Net;
using System.Net.Security;
using System.Reflection;
using System.Security.Cryptography.X509Certificates;
using CSSZSubmissionDemo.Settings;

namespace CSSZSubmissionDemo.Channels
{
    public class SubmissionDataEventArgs : EventArgs
    {
        private byte[] data;
        private string dataId = "";
        private string internalID = "";
        public SubmissionDataEventArgs(string internalID, ref byte[] data, string dataId)
        {
            this.data = data;
            this.dataId = dataId;
            this.internalID = internalID;
        }
        public byte[] Data { get { return data; } }
        public string DataId { get { return dataId; } }
        public string InternalID { get { return internalID; } }
    }

    public class SubmissionStatusEventArgs : EventArgs
    {
        private DAL.SubmissionStatus status;
        private string internalID = "";
        public SubmissionStatusEventArgs(string internalID, DAL.SubmissionStatus status)
        {
            this.status = status;
            this.internalID = internalID;
        }
        public DAL.SubmissionStatus Status { get { return status; } }
        public string InternalID { get { return internalID; } }
    }

    public class SubmissionCreatedEventArgs : EventArgs
    {
        private DAL.SubmissionRecord record;
        public SubmissionCreatedEventArgs(DAL.SubmissionRecord record)
        {
            this.record = record;
        }
        public DAL.SubmissionRecord Record { get { return record; } }
    }

    public delegate void SubmissionDataEventHandler(object sender, SubmissionDataEventArgs e);
    public delegate void SubmissionStatusEventHandler(object sender, SubmissionStatusEventArgs e);
    public delegate void SubmissionCreatedEventHandler(object sender, SubmissionCreatedEventArgs e);
    public delegate void SubmissionResponseEventHandler(object sender, SubmissionDataEventArgs e);

    abstract class ChannelBase
    {
        public const int TIMEOUT = 1200; //in seconds

        public event SubmissionDataEventHandler SubmissionData;
        public event SubmissionStatusEventHandler SubmissionStatus;
        public event SubmissionCreatedEventHandler SubmissionCreated;
        public event SubmissionDataEventHandler SubmissionResponse;

        protected virtual void OnSubmissionData(SubmissionDataEventArgs e)
        {
            if (SubmissionData != null)
            {
                SubmissionData(this, e);
            }
        }

        protected virtual void OnSubmissionStatus(SubmissionStatusEventArgs e)
        {
            if (SubmissionStatus != null)
            {
                SubmissionStatus(this, e);
            }
        }

        protected virtual void OnSubmissionCreated(SubmissionCreatedEventArgs e)
        {
            if (SubmissionCreated != null)
            {
                SubmissionCreated(this, e);
            }
        }

        protected virtual void OnSubmissionResponse(SubmissionDataEventArgs e)
        {
            if (SubmissionResponse != null)
            {
                SubmissionResponse(this, e);
            }
        }

        protected Configuration.ChannelElement settings;
        protected AckInfo ack;

        public AckInfo Ack { get { return ack; } }
        public string DisplayName { get { return settings.DisplayName; } }

        protected void Initialize(Configuration.ChannelElement settings)
        {
            this.settings = settings;
            ServicePointManager.ServerCertificateValidationCallback = new RemoteCertificateValidationCallback(RemoteCertificateValidationCallback);
        }

        static bool RemoteCertificateValidationCallback(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            // Akceptace libovolneho serveroveho certifikatu, neni urceno pro produkcni provoz.
            Trace.WriteLine(String.Format("Server certificate received: Subject={0}, Issuer={1}, SN={2}", ((X509Certificate2)certificate).Subject, ((X509Certificate2)certificate).Issuer, ((X509Certificate2)certificate).SerialNumber));
            return true;
        }

        public static ChannelBase CreateChannel(Configuration.ChannelElement settings)
        {
            try
            {
                ChannelBase c = (ChannelBase)Assembly.GetExecutingAssembly().CreateInstance(settings.ChannelType.FullName);
                if (c != null && settings != null) c.Initialize(settings);
                return c;

            }
            catch (Exception)
            {
                Trace.WriteLine(String.Format("ChannelFactory could not load channel {0}", settings.ChannelType.FullName));
                throw;
            }
        }

        protected abstract string InternalDisplayName();
        protected abstract byte[] GetMinimalMessage(ref Submissions.SubmissionBase submission);

        public abstract void SendSubmission(Submissions.SubmissionBase submission);
        public abstract void PollForResponse(Settings.SubmissionInfo submission);
        public abstract void CompleteTransaction(Settings.SubmissionInfo submission);

        public abstract AckInfo DecodeAck(ref byte[] response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status);
        public abstract AckInfo DecodeAck(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status);
        public abstract AckInfo DecodeResp(ref byte[] response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status);
        public abstract AckInfo DecodeResp(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status);
    }
}
