﻿using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Reflection;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;

namespace CSSZSubmissionDemo.Submissions
{
    abstract class SubmissionBase : MessageBase
    {
        public static SubmissionBase CreateSubmission(string format)
        {
            foreach (var type in Assembly.GetExecutingAssembly().GetTypes().Where(type => type.IsSubclassOf(typeof(SubmissionBase)) && type.IsAbstract == false))
            {
                SubmissionBase s = (SubmissionBase)Assembly.GetExecutingAssembly().CreateInstance(type.FullName);
                //return first channel found
                if (s.InternalDisplayName() == format)
                {
                    return s;
                }
            }
            Trace.WriteLine(String.Format("SubmissionFactory could not create submission format {0}", format));
            return null;
        }

        public abstract byte[] Encode();

        #region HELPER METHODS
        protected byte[] Compress(ref byte[] content)
        {
            Trace.WriteLine("Compressing");
            using (MemoryStream ms = new MemoryStream())
            {
                using (GZipStream gz = new GZipStream(ms, CompressionMode.Compress))
                {
                    gz.Write(content, 0, (int)content.Length);
                    gz.Close(); //IMPORTANT: Close GZipStream before reading compressed result.
                }
                return ms.ToArray();
            }
        }

        protected byte[] Encrypt(ref byte[] content, ref X509Certificate2[] recipients)
        {
            Trace.WriteLine("Encrypting");
            ContentInfo ci = new ContentInfo(content);
            EnvelopedCms cms = new EnvelopedCms(ci);
            CmsRecipientCollection rcps = new CmsRecipientCollection();
            foreach (var recipient in recipients)
            {
                rcps.Add(new CmsRecipient(recipient));
            }
            cms.Encrypt(rcps);
            return cms.Encode();
        }

        protected byte[] Sign(ref byte[] content, ref X509Certificate2 signer)
        {
            Trace.WriteLine("Signing");
            ContentInfo ci = new ContentInfo(content);
            SignedCms scms = new SignedCms(ci, true);
            CmsSigner sgnr = new CmsSigner(signer);
            scms.ComputeSignature(sgnr);
            return scms.Encode();
        }

        protected string Encode64(ref byte[] content)
        {
            Trace.WriteLine("Encoding");
            return Convert.ToBase64String(content);
        }

        #endregion        
    }
}
