﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.ServiceModel;
using System.Text;
using System.Xml;
using CSSZSubmissionDemo.Settings;
using CSSZSubmissionDemo.Submissions;
using CSSZSubmissionDemo.Submissions.WS;

namespace CSSZSubmissionDemo.Channels
{
    class VREPWSChannel : ChannelBase
    {
        protected override string InternalDisplayName()
        {
            return "VREPWS(SOAP)";
        }

        public override void SendSubmission(Submissions.SubmissionBase submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            Trace.WriteLine(string.Format("Encoding submission"));
            byte[] data = GetMinimalMessage(ref submission);
            OnSubmissionData(new SubmissionDataEventArgs(submission.InternalID, ref data, "submit"));

            XmlDocument xBodyData = new XmlDocument();
            xBodyData.LoadXml(Encoding.UTF8.GetString(data));

            BodyPart body = new BodyPart();
            body.Id = "0";
            body.Body = xBodyData.DocumentElement;

            BindingList<BodyPart> bodyList = new BindingList<BodyPart>();
            bodyList.Add(body);

            BindingList<OptionalParameter> optionalPars = new BindingList<OptionalParameter>();
            optionalPars.Add(new OptionalParameter() { ParameterName = "key", ParameterType = "vars", ParameterValue = submission.ID.Value });
            optionalPars.Add(new OptionalParameter() { ParameterName = "timestampversion", ParameterValue = "xmldsig" });

            ServiceClient proxy = CreateProxyClient(((Configuration.VREPWSChannelElement)settings).URI.Anonymous);
            try
            {
                PollResponse resp = proxy.Submit(submission.ClassName, bodyList, optionalPars);
                byte[] respData = Convert.FromBase64String(resp.BodyBase64XML);
                OnSubmissionData(new SubmissionDataEventArgs(submission.InternalID, ref respData, "ack"));

                Trace.WriteLine(string.Format("Decoding ack"));
                DAL.SubmissionStatus s;
                AckInfo ack = this.DecodeAck(ref resp, submission, out s);
                OnSubmissionCreated(new SubmissionCreatedEventArgs(new DAL.SubmissionRecord(this.Ack, this.DisplayName)));
            }
            catch (FaultException vrepEx)
            {
                GGErrorException ggex = vrepEx.CreateMessageFault().GetDetail<GGErrorException>();
                Trace.WriteLine(string.Format("VREP Exception: {0} {1} {2} {3}", ggex.Number, ggex.RaisedBy, ggex.Type, ggex.Text));
            }
            catch (XmlException xmlEx)
            {
                Trace.WriteLine(string.Format("Exception: {0}", xmlEx.ToString()));
            }
            catch (System.ServiceModel.Security.MessageSecurityException mesEx)
            {
                Trace.WriteLine(string.Format("Exception: {0}", mesEx.ToString()));
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception: {0}", ex.ToString()));
            }
        }

        public override void PollForResponse(Settings.SubmissionInfo submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            ServiceClient proxy = CreateProxyClient(((Configuration.VREPWSChannelElement)settings).URI.Anonymous);

            BindingList<OptionalParameter> optionalPars = new BindingList<OptionalParameter>();
            optionalPars.Add(new OptionalParameter() { ParameterName = "key", ParameterType = "vars", ParameterValue = submission.VS });

            try
            {
                Trace.WriteLine(string.Format("Encoding poll"));
                PollResponse resp = proxy.Poll(submission.ID, optionalPars);

                ProcessRequest(proxy, submission.internalID, "poll");
                ProcessResponse(proxy, submission.internalID, "resp");

                Trace.WriteLine(string.Format("Decoding response"));
                DAL.SubmissionStatus s;
                AckInfo ack = this.DecodeAck(ref resp, submission, out s);
                OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, s));
            }
            catch (FaultException vrepEx)
            {
                GGErrorException ggex = vrepEx.CreateMessageFault().GetDetail<GGErrorException>();
                Trace.WriteLine(string.Format("VREP Exception: {0} {1} {2} {3}", ggex.Number, ggex.RaisedBy, ggex.Type, ggex.Text));
            }
            catch (XmlException xmlEx)
            {
                Trace.WriteLine(string.Format("Exception: {0}", xmlEx.ToString()));
            }
            catch (System.ServiceModel.Security.MessageSecurityException mesEx)
            {
                Trace.WriteLine(string.Format("Exception: {0}", mesEx.ToString()));
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception: {0}", ex.ToString()));
            }
        }

        public override void CompleteTransaction(Settings.SubmissionInfo submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            ServiceClient proxy = CreateProxyClient(((Configuration.VREPWSChannelElement)settings).URI.Anonymous);

            BindingList<OptionalParameter> optionalPars = new BindingList<OptionalParameter>();
            optionalPars.Add(new OptionalParameter() { ParameterName = "key", ParameterType = "vars", ParameterValue = submission.VS });

            try
            {
                Trace.WriteLine(string.Format("Encoding delete"));

                proxy.Dispose(submission.ID, optionalPars);
                ProcessRequest(proxy, submission.ID, "poll");
                ProcessResponse(proxy, submission.ID, "close");

                Trace.WriteLine(string.Format("Decoding response"));
                DAL.SubmissionStatus s = DAL.SubmissionStatus.finished;
                OnSubmissionStatus(new SubmissionStatusEventArgs(submission.internalID, s));
            }
            catch (FaultException vrepEx)
            {
                GGErrorException ggex = vrepEx.CreateMessageFault().GetDetail<GGErrorException>();
                Trace.WriteLine(string.Format("VREP Exception: {0} {1} {2} {3}", ggex.Number, ggex.RaisedBy, ggex.Type, ggex.Text));
            }
            catch (System.ServiceModel.Security.MessageSecurityException mesEx)
            {
                Trace.WriteLine(string.Format("Exception: {0}", mesEx.ToString()));
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception: {0}", ex.ToString()));
            }
        }

        protected override byte[] GetMinimalMessage(ref Submissions.SubmissionBase submission)
        {
            Trace.WriteLine(string.Format("Reading settings for channel {0}", InternalDisplayName()));

            XmlDocument doc = new XmlDocument();
            doc.LoadXml(Encoding.UTF8.GetString(submission.Encode()));

            //CSSZ message
            XmlNode nod = doc.SelectSingleNode("/*[local-name()='GovTalkMessage']/*[local-name()='Body']");
            if (nod != null)
            {
                return Encoding.UTF8.GetBytes(nod.InnerXml);
            }
            else
            {
                return null;
            }
        }

        public override Settings.AckInfo DecodeAck(ref byte[] response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            throw new NotImplementedException();
        }

        public override Settings.AckInfo DecodeAck(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            throw new NotImplementedException();
        }

        public Settings.AckInfo DecodeAck(ref PollResponse response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public Settings.AckInfo DecodeAck(ref PollResponse response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }

        public override Settings.AckInfo DecodeResp(ref byte[] response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            throw new NotImplementedException();
        }

        public override Settings.AckInfo DecodeResp(ref byte[] response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            throw new NotImplementedException();
        }

        public Settings.AckInfo DecodeResp(ref PollResponse response, Settings.SubmissionInfo submissionInfo, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submissionInfo.ClassName, submissionInfo.internalID, submissionInfo.VS, out status);
        }

        public Settings.AckInfo DecodeResp(ref PollResponse response, Submissions.SubmissionBase submission, out DAL.SubmissionStatus status)
        {
            return InternalDecode(ref response, submission.ClassName, submission.InternalID, submission.ID.Value, out status);
        }

        private Settings.AckInfo InternalDecode(ref PollResponse response, string className, string internalID, string VS, out DAL.SubmissionStatus status)
        {
            //vsechny odpovedi (ack, response, i error jsou ve formatu GovTalk
            WSResponse wrsp = new WSResponse();
            byte[] foo = new byte[] { };
            wrsp.Initialize(ref foo, new KeyValuePair<string, string>("VS", VS), null, null, null);
            AckInfo a = new AckInfo();
            //GovTalk parser
            byte[] data = wrsp.Decode(ref response, out a);

            status = DAL.SubmissionStatus.created;
            if (a.function.ToLower() == "submit" && (a.qualifier.ToLower() == "response" || (a.qualifier.ToLower() == "error" && string.IsNullOrWhiteSpace(a.ID) == false)))
            {
                status = DAL.SubmissionStatus.sent;
                //pokud jsou v odpovedi obsazena data
                if (data != null) OnSubmissionResponse(new SubmissionDataEventArgs(internalID, ref data, "response.decoded"));
            }
            if (a.function.ToLower() == "delete" && a.qualifier.ToLower() == "response")
            {
                status = DAL.SubmissionStatus.finished;
            }

            ack = a;
            ack.internalID = internalID;
            ack.Class = className;
            ack.VS = VS;
            ack.status = status.ToString();
            return ack;
        }

        private ServiceClient CreateProxyClient(string address)
        {
            WSHttpBinding basicHttp;
            EndpointAddress epAddress;
            ServiceClient proxy;

            // create binding
            basicHttp = new WSHttpBinding(SecurityMode.Transport);
            basicHttp.UseDefaultWebProxy = true;
            basicHttp.Security.Message.EstablishSecurityContext = false;
            basicHttp.Security.Message.NegotiateServiceCredential = false;
            basicHttp.MaxReceivedMessageSize = 512000L;
            basicHttp.ReaderQuotas.MaxStringContentLength = 64000;
            basicHttp.SendTimeout = new TimeSpan(0, Channels.ChannelBase.TIMEOUT, 0);
            basicHttp.OpenTimeout = new TimeSpan(0, Channels.ChannelBase.TIMEOUT, 0);

            // message security type
            basicHttp.Security.Message.ClientCredentialType = MessageCredentialType.None; // anonymous

            // create endpoint
            epAddress = new EndpointAddress(address);

            // create proxy class instance
            proxy = new ServiceClient(basicHttp, epAddress, true);

            return proxy;
        }

        private void ProcessRequest(ServiceClient proxy, string internalId, string dataId)
        {
            string requestText = proxy.Listener.LastRequest;
            if (requestText != null)
            {
                byte[] rawRequest = Encoding.UTF8.GetBytes(requestText);
                OnSubmissionData(new SubmissionDataEventArgs(internalId, ref rawRequest, dataId));
            }
        }

        private void ProcessResponse(ServiceClient proxy, string internalId, string dataId)
        {
            string responseText = proxy.Listener.LastResponse;
            if (responseText != null)
            {
                byte[] rawResponse = Encoding.UTF8.GetBytes(responseText);
                OnSubmissionData(new SubmissionDataEventArgs(internalId, ref rawResponse, dataId));
            }
        }
    }
}
